from django.shortcuts import get_object_or_404, redirect, render
from django.views.generic import ListView, DetailView, TemplateView
from django.contrib.auth.decorators import login_required
from django.utils import timezone
from .models import Quiz, Attempt, Question, Choice, Answer
from django.contrib.auth.mixins import LoginRequiredMixin

class QuizListView(ListView):
    model = Quiz
    template_name = 'quizzes/quiz_list.html'
    context_object_name = 'quizzes'
    paginate_by = 10

    def get_queryset(self):
        return Quiz.objects.filter(is_published=True).order_by('-created_at')

class QuizDetailView(DetailView):
    model = Quiz
    template_name = 'quizzes/quiz_detail.html'
    context_object_name = 'quiz'

@login_required
def start_quiz(request, slug):
    quiz = get_object_or_404(Quiz, slug=slug, is_published=True)
    attempt = Attempt.objects.create(
        user=request.user,
        quiz=quiz,
        started_at=timezone.now(),
        max_score=quiz.questions.count(),
    )
    question_qs = quiz.questions.all().order_by('order')
    if quiz.random_order:
        question_qs = question_qs.order_by('?')
    question_ids = list(question_qs.values_list('id', flat=True))
    request.session[f'attempt_{attempt.id}_question_ids'] = question_ids
    request.session[f'attempt_{attempt.id}_started_at'] = timezone.now().isoformat()
    return redirect('quizzes:attempt_question', attempt_id=attempt.id, question_index=0)

@login_required
def attempt_question(request, attempt_id, question_index):
    attempt = get_object_or_404(Attempt, id=attempt_id, user=request.user)
    key = f'attempt_{attempt.id}_question_ids'
    question_ids = request.session.get(key)
    if question_ids is None:
        return redirect('quizzes:quiz_detail', slug=attempt.quiz.slug)

    if question_index < 0 or question_index >= len(question_ids):
        return redirect('quizzes:finish_attempt', attempt_id=attempt.id)

    question_id = question_ids[question_index]
    question = get_object_or_404(Question, id=question_id)
    choices = question.choices.all()

    if request.method == 'POST':
        selected = request.POST.get('choice')
        choice_obj = None
        if selected:
            try:
                choice_obj = choices.get(id=int(selected))
            except Exception:
                choice_obj = None
        is_correct = bool(choice_obj and choice_obj.is_correct)
        ans, created = Answer.objects.update_or_create(
            attempt=attempt,
            question=question,
            defaults={'choice': choice_obj, 'is_correct': is_correct},
        )
        next_index = question_index + 1
        if next_index >= len(question_ids):
            return redirect('quizzes:finish_attempt', attempt_id=attempt.id)
        return redirect('quizzes:attempt_question', attempt_id=attempt.id, question_index=next_index)

    existing_answer = attempt.answers.filter(question=question).first()
    selected_choice = existing_answer.choice.id if existing_answer and existing_answer.choice else None

    remaining_seconds = None
    if attempt.quiz.time_limit:
        started_iso = request.session.get(f'attempt_{attempt.id}_started_at')
        if started_iso:
            import datetime
            started = timezone.datetime.fromisoformat(started_iso)
            end = started + datetime.timedelta(minutes=attempt.quiz.time_limit)
            remaining = (end - timezone.now()).total_seconds()
            remaining_seconds = int(max(0, remaining))
            if remaining_seconds <= 0:
                return redirect('quizzes:finish_attempt', attempt_id=attempt.id)

    context = {
        'attempt': attempt,
        'question': question,
        'choices': choices,
        'question_index': question_index,
        'total_questions': len(question_ids),
        'selected_choice': selected_choice,
        'remaining_seconds': remaining_seconds,
    }
    return render(request, 'quizzes/attempt_question.html', context)

@login_required
def finish_attempt(request, attempt_id):
    attempt = get_object_or_404(Attempt, id=attempt_id, user=request.user)
    correct = attempt.answers.filter(is_correct=True).count()
    attempt.score = correct
    attempt.finished_at = timezone.now()
    if attempt.started_at:
        attempt.duration = attempt.finished_at - attempt.started_at
    attempt.save()
    request.session.pop(f'attempt_{attempt.id}_question_ids', None)
    request.session.pop(f'attempt_{attempt.id}_started_at', None)
    return render(request, 'quizzes/finish.html', {'attempt': attempt})

class LeaderboardView(ListView):
    model = Attempt
    template_name = 'quizzes/leaderboard.html'
    context_object_name = 'attempts'
    paginate_by = 20

    def get_queryset(self):
        return Attempt.objects.filter(finished_at__isnull=False).order_by('-score', 'finished_at')


class UserHistoryView(LoginRequiredMixin, ListView):
    model = Attempt
    template_name = 'quizzes/history.html'
    context_object_name = 'attempts'
    paginate_by = 20

    def get_queryset(self):
        return Attempt.objects.filter(user=self.request.user).order_by('-finished_at')

@login_required
def attempt_detail(request, attempt_id):
    # ensures only the owner can view
    attempt = get_object_or_404(Attempt, id=attempt_id, user=request.user)

    # all answers for this attempt (select related to avoid extra queries)
    answers = attempt.answers.select_related('question', 'choice').all()

    # total questions for the quiz (use number of quiz questions, not number of answers)
    total_questions = attempt.quiz.questions.count()

    # number correct
    correct_count = sum(1 for a in answers if a.is_correct)

    # We will treat max_score as number of quiz questions (unless attempt.max_score is set)
    max_score = attempt.max_score if attempt.max_score else total_questions
    score = correct_count
    percent = (score / max_score * 100) if max_score else 0

    # collect rows: question, user's choice text, correct choices text, correctness, explanation
    rows = []
    for ans in answers:
        question = ans.question
        user_choice_text = ans.choice.text if ans.choice else None
        # correct choices (could be multiple)
        correct_choices_qs = question.choices.filter(is_correct=True)
        correct_choices_text = ", ".join([c.text for c in correct_choices_qs])
        rows.append({
            'question': question,
            'user_choice_text': user_choice_text,
            'correct_choices_text': correct_choices_text,
            'is_correct': ans.is_correct,
            'explanation': question.explanation,
        })

    # also include questions that were not answered (optional)
    # find unanswered question ids
    answered_q_ids = set(a.question.id for a in answers)
    for q in attempt.quiz.questions.exclude(id__in=answered_q_ids).order_by('order'):
        correct_choices_text = ", ".join([c.text for c in q.choices.filter(is_correct=True)])
        rows.append({
            'question': q,
            'user_choice_text': None,
            'correct_choices_text': correct_choices_text,
            'is_correct': False,
            'explanation': q.explanation,
        })

    context = {
        'attempt': attempt,
        'rows': rows,
        'score': score,
        'max_score': max_score,
        'percent': round(percent, 2),
        'total_questions': total_questions,
        'correct_count': correct_count,
    }
    return render(request, 'quizzes/attempt_detail.html', context)